<?php
/**
 * Admin Panel - Sipariş Listesi
 */
require_once 'config.php';

// Sayfalama ayarları
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Arama ve filtreleme
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$filterDate = isset($_GET['filter_date']) ? trim($_GET['filter_date']) : '';
$filterCity = isset($_GET['filter_city']) ? trim($_GET['filter_city']) : '';

try {
    $pdo = getDBConnection();
    
    // Toplam kayıt sayısı için sorgu
    $countQuery = "SELECT COUNT(*) as total FROM orders WHERE 1=1";
    $countParams = [];
    
    if (!empty($search)) {
        $countQuery .= " AND (
            order_code LIKE :search OR 
            customer_name1 LIKE :search OR 
            customer_surname1 LIKE :search OR
            customer_phone1 LIKE :search OR
            bim_product_name LIKE :search OR
            company_name LIKE :search
        )";
        $countParams[':search'] = "%$search%";
    }
    
    if (!empty($filterDate)) {
        $countQuery .= " AND order_date = :filter_date";
        $countParams[':filter_date'] = $filterDate;
    }
    
    if (!empty($filterCity)) {
        $countQuery .= " AND delivery_city_name = :filter_city";
        $countParams[':filter_city'] = $filterCity;
    }
    
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute($countParams);
    $totalRecords = $countStmt->fetch()['total'];
    $totalPages = ceil($totalRecords / $perPage);
    
    // Sipariş listesi sorgusu
    $query = "SELECT * FROM orders WHERE 1=1";
    $params = [];
    
    if (!empty($search)) {
        $query .= " AND (
            order_code LIKE :search OR 
            customer_name1 LIKE :search OR 
            customer_surname1 LIKE :search OR
            customer_phone1 LIKE :search OR
            bim_product_name LIKE :search OR
            company_name LIKE :search
        )";
        $params[':search'] = "%$search%";
    }
    
    if (!empty($filterDate)) {
        $query .= " AND order_date = :filter_date";
        $params[':filter_date'] = $filterDate;
    }
    
    if (!empty($filterCity)) {
        $query .= " AND delivery_city_name = :filter_city";
        $params[':filter_city'] = $filterCity;
    }
    
    $query .= " ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
    $params[':limit'] = $perPage;
    $params[':offset'] = $offset;
    
    $stmt = $pdo->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value, is_int($value) ? PDO::PARAM_INT : PDO::PARAM_STR);
    }
    $stmt->execute();
    $orders = $stmt->fetchAll();
    
    // Şehir listesi (filtre için)
    $citiesStmt = $pdo->query("SELECT DISTINCT delivery_city_name FROM orders ORDER BY delivery_city_name");
    $cities = $citiesStmt->fetchAll(PDO::FETCH_COLUMN);
    
} catch (Exception $e) {
    $error = $e->getMessage();
    $orders = [];
    $totalPages = 0;
    $cities = [];
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sipariş Yönetim Paneli</title>
    <link rel="stylesheet" href="admin.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>📦 Sipariş Yönetim Paneli</h1>
            <div class="stats">
                <span class="stat-item">Toplam Sipariş: <strong><?php echo number_format($totalRecords); ?></strong></span>
            </div>
        </header>

        <?php if (isset($error)): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <!-- Filtreleme ve Arama -->
        <div class="filters">
            <form method="GET" action="" class="filter-form">
                <div class="filter-group">
                    <input type="text" 
                           name="search" 
                           placeholder="Sipariş kodu, müşteri adı, telefon, ürün adı ara..." 
                           value="<?php echo htmlspecialchars($search); ?>"
                           class="search-input">
                </div>
                
                <div class="filter-group">
                    <input type="date" 
                           name="filter_date" 
                           value="<?php echo htmlspecialchars($filterDate); ?>"
                           class="filter-input"
                           placeholder="Tarih seç">
                </div>
                
                <div class="filter-group">
                    <select name="filter_city" class="filter-input">
                        <option value="">Tüm Şehirler</option>
                        <?php foreach ($cities as $city): ?>
                            <option value="<?php echo htmlspecialchars($city); ?>" 
                                    <?php echo $filterCity === $city ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($city); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <button type="submit" class="btn btn-primary">🔍 Ara</button>
                    <a href="admin.php" class="btn btn-secondary">🔄 Temizle</a>
                </div>
            </form>
        </div>

        <!-- Sipariş Listesi -->
        <div class="orders-table-container">
            <?php if (empty($orders)): ?>
                <div class="alert alert-info">
                    Sipariş bulunamadı.
                </div>
            <?php else: ?>
                <table class="orders-table">
                    <thead>
                        <tr>
                            <th>Sipariş Kodu</th>
                            <th>Tarih</th>
                            <th>Müşteri</th>
                            <th>Telefon</th>
                            <th>Ürün</th>
                            <th>Tutar</th>
                            <th>Şehir</th>
                            <th style="min-width: 150px;">İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orders as $order): ?>
                            <tr>
                                <td>
                                    <strong><?php echo htmlspecialchars($order['order_code']); ?></strong>
                                    <br>
                                    <small>ID: <?php echo $order['order_id']; ?></small>
                                </td>
                                <td>
                                    <?php 
                                    $orderDateTime = new DateTime($order['order_date'] . ' ' . $order['order_time']);
                                    echo $orderDateTime->format('d.m.Y H:i');
                                    ?>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($order['customer_name1'] . ' ' . $order['customer_surname1']); ?>
                                    <br>
                                    <small>TCKN: <?php echo htmlspecialchars($order['customer_tckn']); ?></small>
                                </td>
                                <td><?php echo htmlspecialchars($order['customer_phone1']); ?></td>
                                <td>
                                    <strong><?php echo htmlspecialchars($order['bim_product_name']); ?></strong>
                                    <br>
                                    <small><?php echo htmlspecialchars($order['company_name']); ?></small>
                                </td>
                                <td>
                                    <strong><?php echo number_format($order['total_price'], 2, ',', '.'); ?> ₺</strong>
                                    <br>
                                    <small>Adet: <?php echo $order['order_total_quantity']; ?></small>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($order['delivery_city_name']); ?>
                                    <br>
                                    <small><?php echo htmlspecialchars($order['delivery_district_name']); ?></small>
                                </td>
                                <td>
                                    <div style="display: flex; gap: 5px; flex-wrap: wrap;">
                                        <button class="btn btn-sm btn-info" 
                                                onclick="showOrderDetails(<?php echo htmlspecialchars(json_encode($order)); ?>)">
                                            👁️ Detay
                                        </button>
                                        <button class="btn btn-sm btn-success" 
                                                onclick="sendOrderToAPI(<?php echo $order['id']; ?>, '<?php echo htmlspecialchars($order['order_code']); ?>')"
                                                id="sendBtn_<?php echo $order['id']; ?>">
                                            📤 API'ye Gönder
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Sayfalama -->
                <?php if ($totalPages > 1): ?>
                    <div class="pagination">
                        <?php if ($page > 1): ?>
                            <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&filter_date=<?php echo urlencode($filterDate); ?>&filter_city=<?php echo urlencode($filterCity); ?>" 
                               class="btn btn-sm">« Önceki</a>
                        <?php endif; ?>
                        
                        <span class="page-info">
                            Sayfa <?php echo $page; ?> / <?php echo $totalPages; ?>
                            (<?php echo number_format($totalRecords); ?> kayıt)
                        </span>
                        
                        <?php if ($page < $totalPages): ?>
                            <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&filter_date=<?php echo urlencode($filterDate); ?>&filter_city=<?php echo urlencode($filterCity); ?>" 
                               class="btn btn-sm">Sonraki »</a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>

    <!-- Sipariş Detay Modal -->
    <div id="orderModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal()">&times;</span>
            <div id="orderDetails"></div>
        </div>
    </div>

    <script>
        function showOrderDetails(order) {
            const modal = document.getElementById('orderModal');
            const details = document.getElementById('orderDetails');
            
            // Tarih formatlama
            const orderDate = new Date(order.order_date + 'T' + order.order_time);
            const formattedDate = orderDate.toLocaleDateString('tr-TR', {
                year: 'numeric',
                month: 'long',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
            
            details.innerHTML = `
                <h2>Sipariş Detayları</h2>
                
                <div class="detail-section">
                    <h3>📋 Sipariş Bilgileri</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <label>Sipariş Kodu:</label>
                            <strong>${order.order_code}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Sipariş ID:</label>
                            <strong>${order.order_id}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Tarih:</label>
                            <strong>${formattedDate}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Adet:</label>
                            <strong>${order.order_total_quantity}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Birim Fiyat:</label>
                            <strong>${parseFloat(order.unit_price).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2})} ₺</strong>
                        </div>
                        <div class="detail-item">
                            <label>Toplam Tutar:</label>
                            <strong style="color: #28a745; font-size: 1.2em;">${parseFloat(order.total_price).toLocaleString('tr-TR', {minimumFractionDigits: 2, maximumFractionDigits: 2})} ₺</strong>
                        </div>
                        <div class="detail-item">
                            <label>KDV Oranı:</label>
                            <strong>%${order.tax_rate}</strong>
                        </div>
                        ${order.payment_type ? `<div class="detail-item"><label>Ödeme Tipi:</label><strong>${order.payment_type}</strong></div>` : ''}
                        ${order.bim_invoice_number ? `<div class="detail-item"><label>Fatura No:</label><strong>${order.bim_invoice_number}</strong></div>` : ''}
                    </div>
                </div>

                <div class="detail-section">
                    <h3>🏪 Depo ve Mağaza</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <label>Depo Kodu:</label>
                            <strong>${order.bim_warehouse_code}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Depo Adı:</label>
                            <strong>${order.bim_warehouse_name}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Mağaza Kodu:</label>
                            <strong>${order.bim_store_code}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Mağaza Adı:</label>
                            <strong>${order.bim_store_name}</strong>
                        </div>
                    </div>
                </div>

                <div class="detail-section">
                    <h3>📦 Ürün Bilgileri</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <label>BIM Ürün Kodu:</label>
                            <strong>${order.bim_product_code}</strong>
                        </div>
                        <div class="detail-item">
                            <label>BIM EAN Kodu:</label>
                            <strong>${order.bim_ean_code}</strong>
                        </div>
                        <div class="detail-item full-width">
                            <label>BIM Ürün Adı:</label>
                            <strong>${order.bim_product_name}</strong>
                        </div>
                        ${order.bim_invoice_description ? `<div class="detail-item full-width"><label>Fatura Açıklaması:</label><strong>${order.bim_invoice_description}</strong></div>` : ''}
                        <div class="detail-item">
                            <label>Firma Kodu:</label>
                            <strong>${order.company_code}</strong>
                        </div>
                        <div class="detail-item full-width">
                            <label>Firma Adı:</label>
                            <strong>${order.company_name}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Firma Ürün Kodu:</label>
                            <strong>${order.company_product_code}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Firma EAN Kodu:</label>
                            <strong>${order.company_ean_code}</strong>
                        </div>
                        <div class="detail-item full-width">
                            <label>Firma Ürün Adı:</label>
                            <strong>${order.company_product_name}</strong>
                        </div>
                    </div>
                </div>

                <div class="detail-section">
                    <h3>👤 Müşteri Bilgileri</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <label>TCKN:</label>
                            <strong>${order.customer_tckn}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Ad:</label>
                            <strong>${order.customer_name1}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Soyad:</label>
                            <strong>${order.customer_surname1}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Telefon:</label>
                            <strong>${order.customer_phone1}</strong>
                        </div>
                        ${order.customer_name2 ? `
                            <div class="detail-item">
                                <label>2. Kişi Ad:</label>
                                <strong>${order.customer_name2}</strong>
                            </div>
                            <div class="detail-item">
                                <label>2. Kişi Soyad:</label>
                                <strong>${order.customer_surname2}</strong>
                            </div>
                            <div class="detail-item">
                                <label>2. Kişi Telefon:</label>
                                <strong>${order.customer_phone2}</strong>
                            </div>
                        ` : ''}
                    </div>
                </div>

                <div class="detail-section">
                    <h3>🚚 Teslimat Adresi</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <label>İl:</label>
                            <strong>${order.delivery_city_name}</strong>
                        </div>
                        <div class="detail-item">
                            <label>İlçe:</label>
                            <strong>${order.delivery_district_name}</strong>
                        </div>
                        <div class="detail-item">
                            <label>Mahalle:</label>
                            <strong>${order.delivery_neighbourhood_name}</strong>
                        </div>
                        <div class="detail-item full-width">
                            <label>Adres 1:</label>
                            <strong>${order.delivery_address1}</strong>
                        </div>
                        ${order.delivery_address2 ? `<div class="detail-item full-width"><label>Adres 2:</label><strong>${order.delivery_address2}</strong></div>` : ''}
                    </div>
                </div>

                <div class="detail-section">
                    <h3>⏰ Sistem Bilgileri</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <label>Oluşturulma:</label>
                            <strong>${new Date(order.created_at).toLocaleString('tr-TR')}</strong>
                        </div>
                        ${order.updated_at && order.updated_at !== order.created_at ? `
                            <div class="detail-item">
                                <label>Güncellenme:</label>
                                <strong>${new Date(order.updated_at).toLocaleString('tr-TR')}</strong>
                            </div>
                        ` : ''}
                    </div>
                </div>
            `;
            
            modal.style.display = 'block';
        }

        function closeModal() {
            document.getElementById('orderModal').style.display = 'none';
        }

        // Modal dışına tıklanınca kapat
        window.onclick = function(event) {
            const modal = document.getElementById('orderModal');
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        }

        // API'ye sipariş gönder
        function sendOrderToAPI(orderId, orderCode) {
            const btn = document.getElementById('sendBtn_' + orderId);
            const originalText = btn.innerHTML;
            
            if (!confirm('Bu siparişi e-ticaret sistemine göndermek istediğinize emin misiniz?')) {
                return;
            }
            
            btn.disabled = true;
            btn.innerHTML = '⏳ Gönderiliyor...';
            
            fetch('send_order_to_api.php?id=' + orderId, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    btn.innerHTML = '✅ Gönderildi';
                    btn.style.background = '#28a745';
                    alert('Sipariş başarıyla e-ticaret sistemine gönderildi!\n\nSipariş Kodu: ' + (data.order_code || orderCode));
                } else {
                    btn.innerHTML = originalText;
                    btn.disabled = false;
                    alert('Hata: ' + (data.message || 'Sipariş gönderilemedi'));
                }
            })
            .catch(error => {
                btn.innerHTML = originalText;
                btn.disabled = false;
                alert('Hata: ' + error.message);
            });
        }
    </script>
</body>
</html>

