<?php
/**
 * Order Save Handler
 * Receives JSON order data and saves it to the database
 */

require_once 'config.php';

header('Content-Type: application/json; charset=utf-8');

/**
 * Save Order to Database
 * @param array $orderData
 * @param PDO $pdo
 * @return array
 */
function saveOrder($orderData, $pdo) {
    try {
        // Parse order date and time
        $orderDate = DateTime::createFromFormat('d.m.Y', $orderData['orderDate']);
        if (!$orderDate) {
            throw new Exception("Invalid order date format");
        }
        $orderTime = DateTime::createFromFormat('H:i:s', $orderData['orderTime']);
        if (!$orderTime) {
            throw new Exception("Invalid order time format");
        }
        
        // Tek bir INSERT sorgusu ile tüm bilgileri kaydet
        $stmt = $pdo->prepare("
            INSERT INTO orders (
                order_id, order_code, order_date, order_time,
                order_total_quantity, unit_price, total_price, tax_rate,
                payment_type, bim_invoice_number,
                bim_warehouse_code, bim_warehouse_name,
                bim_store_code, bim_store_name,
                bim_product_code, bim_ean_code, bim_product_name,
                bim_invoice_description,
                company_code, company_name,
                company_product_code, company_ean_code, company_product_name,
                customer_tckn, customer_name1, customer_surname1, customer_phone1,
                customer_name2, customer_surname2, customer_phone2,
                delivery_city_name, delivery_district_name,
                delivery_neighbourhood_name, delivery_address1, delivery_address2
            ) VALUES (
                :order_id, :order_code, :order_date, :order_time,
                :order_total_quantity, :unit_price, :total_price, :tax_rate,
                :payment_type, :bim_invoice_number,
                :bim_warehouse_code, :bim_warehouse_name,
                :bim_store_code, :bim_store_name,
                :bim_product_code, :bim_ean_code, :bim_product_name,
                :bim_invoice_description,
                :company_code, :company_name,
                :company_product_code, :company_ean_code, :company_product_name,
                :customer_tckn, :customer_name1, :customer_surname1, :customer_phone1,
                :customer_name2, :customer_surname2, :customer_phone2,
                :delivery_city_name, :delivery_district_name,
                :delivery_neighbourhood_name, :delivery_address1, :delivery_address2
            )
        ");
        
        $stmt->execute([
            // Sipariş Temel Bilgileri
            ':order_id' => $orderData['orderId'],
            ':order_code' => $orderData['orderCode'],
            ':order_date' => $orderDate->format('Y-m-d'),
            ':order_time' => $orderTime->format('H:i:s'),
            ':order_total_quantity' => $orderData['orderTotalQuantity'],
            ':unit_price' => $orderData['unitPrice'],
            ':total_price' => $orderData['totalPrice'],
            ':tax_rate' => $orderData['taxRate'],
            ':payment_type' => !empty($orderData['paymentType']) ? $orderData['paymentType'] : null,
            ':bim_invoice_number' => !empty($orderData['bimInvoiceNumber']) ? $orderData['bimInvoiceNumber'] : null,
            // Depo ve Mağaza Bilgileri
            ':bim_warehouse_code' => $orderData['bimWareHouseCode'],
            ':bim_warehouse_name' => $orderData['bimWareHouseName'],
            ':bim_store_code' => $orderData['bimStoreCode'],
            ':bim_store_name' => $orderData['bimStoreName'],
            // Ürün Bilgileri (BIM)
            ':bim_product_code' => $orderData['bimProductCode'],
            ':bim_ean_code' => $orderData['bimEanCode'],
            ':bim_product_name' => $orderData['bimProductName'],
            ':bim_invoice_description' => !empty($orderData['bimInvoiceDescription']) ? $orderData['bimInvoiceDescription'] : null,
            // Firma Bilgileri
            ':company_code' => $orderData['companyCode'],
            ':company_name' => $orderData['companyName'],
            ':company_product_code' => $orderData['companyProductCode'],
            ':company_ean_code' => $orderData['companyEanCode'],
            ':company_product_name' => $orderData['companyProductName'],
            // Müşteri Bilgileri
            ':customer_tckn' => $orderData['customerTckn'],
            ':customer_name1' => $orderData['customerName1'],
            ':customer_surname1' => $orderData['customerSurname1'],
            ':customer_phone1' => $orderData['customerPhone1'],
            ':customer_name2' => !empty($orderData['customerName2']) ? $orderData['customerName2'] : null,
            ':customer_surname2' => !empty($orderData['customerSurname2']) ? $orderData['customerSurname2'] : null,
            ':customer_phone2' => !empty($orderData['customerPhone2']) ? $orderData['customerPhone2'] : null,
            // Teslimat Adresi Bilgileri
            ':delivery_city_name' => $orderData['deliveryCityName'],
            ':delivery_district_name' => $orderData['deliveryDistrictName'],
            ':delivery_neighbourhood_name' => $orderData['deliveryNeighbourhoodName'],
            ':delivery_address1' => $orderData['deliveryAddress1'],
            ':delivery_address2' => !empty($orderData['deliveryAddress2']) ? $orderData['deliveryAddress2'] : null
        ]);
        
        return [
            'success' => true,
            'message' => 'Order saved successfully',
            'order_code' => $orderData['orderCode']
        ];
        
    } catch (PDOException $e) {
        throw new Exception("Database error: " . $e->getMessage());
    } catch (Exception $e) {
        throw $e;
    }
}

// Main execution
try {
    // Get JSON input
    $jsonInput = file_get_contents('php://input');
    
    if (empty($jsonInput)) {
        // Try to get from POST data if available
        if (isset($_POST['order_data'])) {
            $jsonInput = $_POST['order_data'];
        } elseif (isset($_FILES['order_file'])) {
            // If uploaded as file
            $jsonInput = file_get_contents($_FILES['order_file']['tmp_name']);
        } else {
            throw new Exception("No JSON data received");
        }
    }
    
    // Decode JSON
    $orderData = json_decode($jsonInput, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Invalid JSON format: " . json_last_error_msg());
    }
    
    // Validate required fields
    $requiredFields = [
        'orderId', 'orderCode', 'orderDate', 'orderTime',
        'orderTotalQuantity', 'unitPrice', 'totalPrice', 'taxRate',
        'bimWareHouseCode', 'bimWareHouseName', 'bimStoreCode', 'bimStoreName',
        'bimProductCode', 'bimEanCode', 'bimProductName',
        'companyCode', 'companyName', 'companyProductCode', 'companyEanCode', 'companyProductName',
        'customerTckn', 'customerName1', 'customerSurname1', 'customerPhone1',
        'deliveryCityName', 'deliveryDistrictName', 'deliveryNeighbourhoodName', 'deliveryAddress1'
    ];
    
    foreach ($requiredFields as $field) {
        if (!isset($orderData[$field])) {
            throw new Exception("Required field missing: " . $field);
        }
    }
    
    // Get database connection
    $pdo = getDBConnection();
    
    // Check if order already exists
    $stmt = $pdo->prepare("SELECT id FROM orders WHERE order_code = :order_code");
    $stmt->execute([':order_code' => $orderData['orderCode']]);
    if ($stmt->fetch()) {
        throw new Exception("Order with code " . $orderData['orderCode'] . " already exists");
    }
    
    // Save order
    $result = saveOrder($orderData, $pdo);
    
    http_response_code(200);
    echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
}

